document.addEventListener('DOMContentLoaded', function() {
    
    // --- 1. HERO STATS ANIMATION ---
    const counters = document.querySelectorAll('.stat-value');
    const speed = 200; // The lower the slower

    const animateCounters = () => {
        counters.forEach(counter => {
            const updateCount = () => {
                const target = +counter.getAttribute('data-target');
                const count = +counter.innerText;
                const inc = target / speed;

                if (count < target) {
                    counter.innerText = Math.ceil(count + inc);
                    setTimeout(updateCount, 20);
                } else {
                    counter.innerText = target;
                    // Add % if needed
                    if(target === 3) counter.innerText = target + '%';
                }
            };
            updateCount();
        });
    };
    
    // Trigger animation when hero is visible
    animateCounters();

    // --- 2. TABS LOGIC ---
    const tabBtns = document.querySelectorAll('.tab-btn[data-tab]');
    const tabContents = document.querySelectorAll('.tab-content');

    tabBtns.forEach(btn => {
        btn.addEventListener('click', () => {
            // Remove active class from all
            tabBtns.forEach(b => b.classList.remove('active'));
            tabContents.forEach(c => c.classList.remove('active'));

            // Add active to current
            btn.classList.add('active');
            document.getElementById(btn.getAttribute('data-tab')).classList.add('active');
        });
    });

    // --- 3. DATATABLES ---
    const commonTableOptions = {
        paging: true,
        searching: true,
        info: false,
        language: {
            url: '//cdn.datatables.net/plug-ins/1.13.6/i18n/fr-FR.json'
        }
    };

    // Table France
    $('#table_france').DataTable({
        ...commonTableOptions,
        data: [
            ['Centres de données', '460', '2023', 'Identifiés ≥ 1 GWh/an'],
            ['Consommation électrique', '3.9 TWh', '2023', 'Centres > 1 GWh/an'],
            ['Concentration IdF', '64 %', '2023', 'Part consommation nationale'],
            ['Croissance conso.', '+21 %', '2022-2023', 'Gros centres'],
            ['Part mondiale', '~1.5 %', '2024', 'Estimation']
        ]
    });

    // Table Monde
    $('#table_monde').DataTable({
        ...commonTableOptions,
        data: [
            ['Conso. Data Centers', '415 TWh', '2024', 'IEA'],
            ['Projection', '945 TWh', '2030', 'Scénario médian'],
            ['Part élec. mondiale', '1.5 %', '2022', 'IEA'],
            ['Part élec. mondiale', '3 %', '2030', 'Projection'],
            ['Croissance annuelle', '+4.3 %', '2024', 'Estimation']
        ]
    });

    // Table IA
    $('#table_ia').DataTable({
        ...commonTableOptions,
        data: [
            ['Conso. IA Générative', '5.7 - 8.9 TWh', '2023', 'Estimation'],
            ['Requête ChatGPT', '2.9 Wh', '-', 'x10 vs Google'],
            ['Entraînement GPT-3', '1 287 MWh', '-', 'Éq. 120 ans conso foyer'],
            ['Projection IA', '134 TWh', '2027', 'Scénario haut']
        ]
    });

    // --- 4. VISUALIZATIONS ---
    
    // Strict: Chart.js
    const ctxStrict = document.getElementById('chart_strict');
    if (ctxStrict) {
        new Chart(ctxStrict, {
            type: 'bar',
            data: {
                labels: ['2018', '2019', '2020', '2021', '2022', '2023'],
                datasets: [{
                    label: 'Consommation (TWh)',
                    data: [12, 14, 16, 18, 20, 22], 
                    backgroundColor: '#64FFDA',
                    borderColor: '#64FFDA',
                    borderWidth: 1
                },
                {
                    type: 'line',
                    label: 'Tendance',
                    data: [12, 14, 16, 18, 20, 22],
                    borderColor: '#FF6B6B',
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { labels: { color: '#E6F1FF' } }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: { color: '#233554' },
                        ticks: { color: '#A8B2D1' }
                    },
                    x: {
                        grid: { display: false },
                        ticks: { color: '#A8B2D1' }
                    }
                }
            }
        });
    }

    // Hybrid: SVG Animation
    const animateHybridGraph = () => {
        const barColo = document.getElementById('bar_colo');
        const barEnt = document.getElementById('bar_ent');
        const barCloud = document.getElementById('bar_cloud');
        const textColo = document.getElementById('text_colo');
        const textEnt = document.getElementById('text_ent');
        const textCloud = document.getElementById('text_cloud');

        if (barColo && barEnt && barCloud) {
            const startY = 419.22; // Baseline Y from SVG
            
            const targets = [
                { el: barColo, h: +barColo.getAttribute('data-height'), y: +barColo.getAttribute('data-y') },
                { el: barEnt, h: +barEnt.getAttribute('data-height'), y: +barEnt.getAttribute('data-y') },
                { el: barCloud, h: +barCloud.getAttribute('data-height'), y: +barCloud.getAttribute('data-y') }
            ];

            let progress = 0;
            const duration = 1500; // ms
            const startTime = performance.now();

            const easeOutQuart = (x) => 1 - Math.pow(1 - x, 4);

            const animate = (currentTime) => {
                const elapsed = currentTime - startTime;
                progress = Math.min(elapsed / duration, 1);
                const ease = easeOutQuart(progress);

                targets.forEach(t => {
                    const currentH = t.h * ease;
                    const currentY = startY - currentH;
                    t.el.setAttribute('height', currentH);
                    t.el.setAttribute('y', currentY);
                });

                if (progress < 1) {
                    requestAnimationFrame(animate);
                } else {
                    // Fade in text
                    [textColo, textEnt, textCloud].forEach(txt => {
                        txt.style.transition = 'opacity 0.5s';
                        txt.style.opacity = '1';
                    });
                }
            };
            requestAnimationFrame(animate);
        }
    };

    // Trigger animation when in view
    const hybridSection = document.querySelector('.hybrid-wrap');
    if(hybridSection) {
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if(entry.isIntersecting) {
                    animateHybridGraph();
                    observer.unobserve(entry.target);
                }
            });
        }, { threshold: 0.5 });
        observer.observe(hybridSection);
    }

    // Creative: SVG Animation (World 2024)
    const animateCreativeGraph = () => {
        const segments = ['seg_reste', 'seg_na', 'seg_asia', 'seg_eu'];
        const labels = document.getElementById('creative_labels');
        const centerText = document.getElementById('center_text_creative');

        // Reset
        if(labels) labels.style.opacity = '0';
        if(centerText) centerText.style.opacity = '0';

        segments.forEach(id => {
            const path = document.getElementById(id);
            if(path) {
                const length = path.getTotalLength();
                // Set up dash array for drawing effect
                path.style.strokeDasharray = length;
                path.style.strokeDashoffset = length;
                path.style.fillOpacity = '0';
                path.style.transition = 'none';
                
                // Trigger reflow
                path.getBoundingClientRect();

                // Animate Stroke
                path.style.transition = 'stroke-dashoffset 1.5s ease-in-out, fill-opacity 0.5s ease-in-out 1.5s';
                path.style.strokeDashoffset = '0';
                path.style.fillOpacity = '1';
            }
        });

        // Animate Labels & Text
        setTimeout(() => {
            if(labels) {
                labels.style.transition = 'opacity 1s ease';
                labels.style.opacity = '1';
            }
            if(centerText) {
                centerText.style.transition = 'opacity 1s ease';
                centerText.style.opacity = '1';
            }
        }, 1500);
    };

    // Observer for auto-play on scroll
    const creativeSection = document.querySelector('.creative-viz');
    if(creativeSection) {
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if(entry.isIntersecting) {
                    animateCreativeGraph();
                    observer.unobserve(entry.target);
                }
            });
        }, { threshold: 0.5 });
        observer.observe(creativeSection);
    }

    // Play button logic
    const playBtn = document.getElementById('play_creative');
    if(playBtn) {
        playBtn.addEventListener('click', () => {
            // Reset first to allow re-animation
            const segments = ['seg_reste', 'seg_na', 'seg_asia', 'seg_eu'];
            segments.forEach(id => {
                const path = document.getElementById(id);
                if(path) {
                    path.style.transition = 'none';
                    path.style.strokeDashoffset = path.getTotalLength();
                    path.style.fillOpacity = '0';
                }
            });
            const labels = document.getElementById('creative_labels');
            if(labels) labels.style.opacity = '0';
            const centerText = document.getElementById('center_text_creative');
            if(centerText) centerText.style.opacity = '0';

            // Small delay to let browser register reset
            setTimeout(() => {
                animateCreativeGraph();
            }, 50);
        });
    }


    // --- 5. LIGHTBOX ---
    window.openLightbox = (btn) => {
        const img = btn.parentElement.previousElementSibling;
        const lightbox = document.getElementById('lightbox');
        const lightboxImg = document.getElementById('lightbox-img');
        lightboxImg.src = img.src;
        lightbox.classList.add('active');
    };

    window.closeLightbox = () => {
        document.getElementById('lightbox').classList.remove('active');
    };

});

